import codecs
import importlib.util
import inspect
import pickle
import sys
from enum import Enum
from peewee import *
from core.db import db_proxy
from core.models import Proxy


class ClusterSettings(Model):

    class SettingDefaults(Enum):
        ERROR_STORAGE_MACHINE = "ERROR_STORAGE_MACHINE"
        ERROR_SCREENSHOTS_DIR = "ERROR_SCREENSHOTS_DIR"
        ERROR_DOM_DIR = "ERROR_DOM_DIR"

    class Meta:
        database = db_proxy

    setting_name = CharField(max_length=32)
    setting_value = CharField(max_length=512)

    @staticmethod
    def get_value_of(setting_name: SettingDefaults):
        return ClusterSettings.get(setting_name=setting_name.name).setting_value


class Settings:

    def __init__(self, base_url, queries_per_second, max_workers, outputs, context_class, init_function,
                 languages=['en'], inputs=[], debug=False, celery_broker_url=None, layers=[], max_retries=3):
        self.base_url = base_url
        self.queries_per_second = queries_per_second
        self.outputs = {}
        self.inputs = {}
        try:
            self.proxies = list(Proxy.select().where(Proxy.is_active == 1))
        except Exception as ex:
            self.proxies = []
            print(ex)

        self.context_class = context_class
        self.init_function = init_function
        self.languages = ','.join(languages)
        self.inputs = {}
        self.debug = debug
        self.celery_broker_url = celery_broker_url
        self.layers = layers
        self.max_retries = max_retries

        if max_workers > len(self.proxies):
            print(f"There are not enough proxies. Reduced number of workers to {len(self.proxies)}")

        self.max_workers = max_workers if len(self.proxies) >= max_workers else len(self.proxies)

        for output in outputs:
            self.outputs[output.code] = output

        for input_ in inputs:
            self.inputs[input_.code] = input_

    def get_unlocked_proxy(self):
        for proxy in self.proxies:
            if not proxy.is_locked():
                proxy.lock()
                return proxy

    def enable_debug(self):
        self.debug = True

    def disable_debug(self):
        self.debug = False

    def disable_partitioning(self):
        for layer in self.layers:
            layer.partition = False

    def close(self):
        for output in self.outputs.values():
            output.close()

        for input in self.inputs.values():
            input.close()

    def create_remote_settings(self):
        remote_settings = RemoteSettings(base_url=self.base_url,
                                queries_per_second=self.queries_per_second,
                                max_workers=self.max_workers,
                                outputs=[],
                                inputs=[],
                                context_class=self.context_class,
                                init_function=self.init_function,
                                debug=self.debug,
                                layers=self.layers,
                                max_retries=self.max_retries)
        remote_settings.dump_modules()
        return remote_settings


class RemoteSettings(Settings):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.dumped_modules = {}

    def dump_modules(self):

        for module_name, module in sys.modules.items():
            if not module_name.startswith("crawlers.") or "_main" in module_name:
                continue
            if module_name not in self.dumped_modules:
                with open(module.__file__, 'r') as file:
                    self.dumped_modules[module_name] = file.read()

        self.init_function = codecs.encode(pickle.dumps(self.init_function), "base64").decode()
        self.layers = codecs.encode(pickle.dumps(self.layers), "base64").decode()

    def load_modules(self):
        for module_name, source in self.dumped_modules.items():
            self.import_module_from_string(name=module_name, source=source)

        self.init_function = pickle.loads(codecs.decode(self.init_function.encode(), "base64"))
        self.layers = pickle.loads(codecs.decode(self.layers.encode(), "base64"))

    @staticmethod
    def import_module_from_string(name: str, source: str):
        """
        Import module from source string.
        Example use:
        import_module_from_string("m", "f = lambda: print('hello')")
        m.f()
        """
        spec = importlib.util.spec_from_loader(name, loader=None)
        module = importlib.util.module_from_spec(spec)
        exec(source, module.__dict__)
        sys.modules[name] = module
        globals()[name] = module
