import os
import time
import psutil
from functools import wraps

from selenium.common import TimeoutException
from selenium.webdriver.support.wait import WebDriverWait



def timeit(func):
    @wraps(func)
    def timeit_wrapper(*args, **kwargs):
        start_time = time.perf_counter()
        result = func(*args, **kwargs)
        end_time = time.perf_counter()
        total_time = end_time - start_time
        print(f'Function {func.__name__}{args} {kwargs} Took {total_time:.4f} seconds')
        return result
    return timeit_wrapper


class TerminatedException(Exception):
    pass


class ResourceController:

    UPDATE_FREQUENCY = 20

    SUSPEND_MARGIN = 1.5
    AFTER_SUSPEND_DROP = 1.2

    REGULATING_MARGIN = 3
    REGULATING_COEFFICIENT = 0.2

    GB_CONVERT = 2**30

    USAGE_FILE_PATH = {
        1: "/sys/fs/cgroup/memory/memory.usage_in_bytes",
        2: "/sys/fs/cgroup/memory.current"
    }
    
    LIMIT_FILE_PATH = {
        1: "/sys/fs/cgroup/memory/memory.limit_in_bytes",
        2: "/sys/fs/cgroup/memory.max"
    }

    def __init__(self):
        self.memory_limit = self.get_memory_limit()
        self.current_memory_usage = self.get_current_memory_usage()
        self.memory_usage_of_process, self.cpu_usage_of_process = self.get_usage_of_process()

        self.last_update = time.time()

    def control(self) -> None:
        self.count_resources_usage()
        total_sleep = 0

        if self.get_free_memory() <= self.REGULATING_MARGIN:
             total_sleep += self.REGULATING_COEFFICIENT * ( self.memory_usage_of_process + 1 / self.get_free_memory() )


        if 2*self.cpu_usage_of_process > 1.5:
            total_sleep += 1.5
        else:
            total_sleep += 2*self.cpu_usage_of_process

        if total_sleep > 3:
            total_sleep = 3

        time.sleep(total_sleep)

    def count_resources_usage(self):
        if time.time() - self.last_update >= self.UPDATE_FREQUENCY:
            self.current_memory_usage = self.get_current_memory_usage()
            self.memory_usage_of_process, self.cpu_usage_of_process = self.get_usage_of_process()

            self.last_update = time.time()

    def get_free_memory(self):
        return self.memory_limit - self.current_memory_usage

    @classmethod
    def get_memory_limit(cls) -> float:
        with open(cls.LIMIT_FILE_PATH[cls.get_cgroup_version()]) as limit_in_bytes_file:
            limit_in_bytes = limit_in_bytes_file.read()
        try:
            return int(limit_in_bytes) / cls.GB_CONVERT  # convert bytes to GB
        except:
            return 16

    @classmethod
    def get_current_memory_usage(cls) -> float:
        with open(cls.USAGE_FILE_PATH[cls.get_cgroup_version()], 'r') as usage_in_bytes_file:
            usage_in_bytes = usage_in_bytes_file.read()
        return int(usage_in_bytes) / cls.GB_CONVERT  # convert bytes to GB

    @classmethod
    @timeit
    def get_usage_of_process(cls) -> float:
        current_process = psutil.Process(os.getpid())
        memory_usage = current_process.memory_info().rss
        cpu_usage = current_process.cpu_percent(interval=0.2)
        for child in current_process.children(recursive=True):
            try:
                memory_usage += child.memory_info().rss
                cpu_usage += child.cpu_percent(interval=0.2)
            except Exception as ex:
                print(ex)

        return memory_usage / cls.GB_CONVERT, cpu_usage / 100

    @classmethod
    def get_cgroup_version(cls) -> int:
        try:
            with open("/sys/fs/cgroup/cgroup.controllers", 'r') as controllers:
                return 2
        except Exception as ex:
            return 1


methods_execution_time = {}


class CustomWebDriverWait(WebDriverWait):

    def __init__(self, *args, **kwargs):
        super().__init__(*args,  **kwargs, poll_frequency=2)
        self.name = ""
        self.waited = False
        self.reached_threshhold = False

    def until(self, method, name: str, message: str = ""):
        screen = None
        stacktrace = None

        self.name = name

        end_time = time.monotonic() + self._timeout
        start_time = time.time()
        while True:
            try:
                value = method(self._driver)
                if value:
                    methods_execution_time[name] = self.get_next_wait_time(time.time() - start_time)
                    return value
            except self._ignored_exceptions as exc:
                screen = getattr(exc, "screen", None)
                stacktrace = getattr(exc, "stacktrace", None)

            self.sleep()

            if time.monotonic() > end_time:
                break
        raise TimeoutException(message, screen, stacktrace)

    def sleep(self):

        if methods_execution_time.get(self.name) and not self.waited:
            self.waited = True
            time.sleep(methods_execution_time[self.name])
        else:
            time.sleep(self._poll)

    @classmethod
    def get_next_wait_time(cls, current_waiting):
        if current_waiting > 10:
            return 1
        else:
            return current_waiting

