from __future__ import annotations

import time
import hashlib

from core.models import ErrorNode


class Node:

    def __init__(self, name=None, parent: Node = None, page_url=None, page_content=None, node_id=None, layer=0):
        self.name = name
        self.node_id = node_id if node_id else hashlib.md5(self.name.encode()).hexdigest()
        self.page_url = page_url
        self.page_content = page_content
        self.properties = {}
        self.layer = layer

        self.parent = parent
        self.children = []

        self.locked = False

        self.cached_ancestor_chain = []

        self.__clean_name()

    def __str__(self):
        return f"<Node: {self.name}>"

    def __repr__(self):
        return str(self)

    def pop(self):
        if self.parent:
            self.parent.remove_child(self)

    def add_children(self, name, page_url=None, page_content=None, node_id=None) -> Node:
        if page_url:
            new_node = Node(name=name, page_url=page_url, node_id=node_id)
        else:
            new_node = Node(name=name, page_content=page_content)
        new_node.parent = self
        new_node.layer = self.layer + 1
        self.children.append(new_node)
        return new_node

    def remove_child(self, child):
        if child not in self.children:
            return
        self.children.remove(child)
        if not self.children and self.parent:
            self.parent.remove_child(self)

    def fetch_content(self, context):

        page = context.make_request(self.page_url)
        self.page_content = page
        # if not page:
        #     self.page_content = BeautifulSoup("", "html.parser")
        #     return
        #
        # self.page_content = BeautifulSoup(page.content, "html.parser")
        context.last_fetch = time.time()

    def cache_ancestor_chain(self):
        self.cached_ancestor_chain = self.get_ancestor_chain()

    def get_ancestor_chain(self):
        if not self.parent:
            return []
        else:
            if self.cached_ancestor_chain:
                return self.cached_ancestor_chain
            else:
                return self.parent.get_ancestor_chain() + [self]

    def get_ancestor_path(self):
        return "/".join(map(lambda x: x.name, self.get_ancestor_chain()))

    def lock(self):
        self.locked = True

    def unlock(self):
        self.locked = False

    def is_locked(self):
        return self.locked

    def __clean_name(self):
        self.name = self.name.replace("/", " ")


class Tree:

    def __init__(self):
        self.root = None

    def add_root_node(self, name, page_url):
        self.root = Node(name=name, page_url=page_url)
        return self.root

    def set_root_node(self, node):
        self.root = node
