﻿from distutils.version import LooseVersion
from selenium import webdriver
from webdriver_manager.chrome import ChromeDriverManager
import os
import platform
import re
import zipfile


CHROME_BINARY_DRIVER = "D:/www/google-map/drivers/chromedriver.exe"
CHROME_PLUGIN_FILE = 'proxy-chrome.xpi'
RESPONSE_TIMEOUT = 10
PROXY_REGEXP = '^(?P<proto>https?://)?(?P<usr>[^:]+):(?P<pwd>[^@]+)@(?P<host>[^:]+):(?P<port>\d+)$'

# normal	complete	Used by default, waits for all resources to download
# eager	    	interactive	DOM access is ready, but other resources like images may still be loading
# none	    	Any	        Does not block WebDriver at all
STRATEGY = "normal"         	# "normal", "eager", "none"


def chrome_manifest():
    return """
    {
        "version": "1.0.0",
        "manifest_version": 2,
        "name": "My Chrome Proxy",
        "permissions": [
            "proxy",
            "tabs",
            "unlimitedStorage",
            "storage",
            "<all_urls>",
            "webRequest",
            "webRequestBlocking"
        ],
        "background": { "scripts": ["background.js"]}, 
        "minimum_chrome_version":"22.0.0"
    }
    """


def chrome_background_js(**kwargs):
    return """
    let config = {
            mode: "fixed_servers",
            rules: {
            singleProxy: {
                scheme: "http",
                host: "%(host)s",
                port: %(port)s
            },
            bypassList: ["localhost"]
            }
        };

    chrome.proxy.settings.set({value: config, scope: "regular"}, function() {});

    function callbackFn(details) {
        return {
            authCredentials: {
                username: "%(user)s",
                password: "%(password)s"
            }
        };
    }

    chrome.webRequest.onAuthRequired.addListener(
                callbackFn,
                {urls: ["<all_urls>"]},
                ['blocking']
    );
    """ % kwargs


def get_latest_chromedriver_path():
    try:
        home_dir = os.path.expanduser("~")
        if platform.system() == 'Windows':
            sub_path = os.path.join('drivers', 'chromedriver', 'win32')
            driver_name = 'chromedriver.exe'
        else:  # Assuming Unix-based system
            sub_path = os.path.join('drivers', 'chromedriver', 'linux64')
            driver_name = 'chromedriver'
        path = os.path.join(home_dir, '.wdm', sub_path)
        subfolders = [f.path for f in os.scandir(path) if f.is_dir()]
        # Extract versions from the folder names
        versions = [folder.split(os.sep)[-1] for folder in subfolders]
        # Compare versions and find the latest
        latest_version = str(max(LooseVersion(ver) for ver in versions))
        # Full path to the chromedriver with the latest version
        latest_chromedriver_path = os.path.join(path, latest_version, driver_name)
    except FileNotFoundError:
        latest_chromedriver_path = None
    return latest_chromedriver_path or ChromeDriverManager().install()


def get_chromedriver(proxy='', user_agent=None, strategy=STRATEGY, debug=False):
    chrome_options = webdriver.ChromeOptions()
    chrome_options.add_argument('--disable-gpu')
    chrome_options.add_argument('--disable-dev-shm-usage')
    chrome_options.add_argument('--disable-audio')
    chrome_options.add_argument('--no-sandbox')
    chrome_options.add_argument('--ignore-certificate-errors')
    chrome_options.page_load_strategy = strategy
    if not debug:
        chrome_options.add_argument('--headless')
    use_proxy = re.match(PROXY_REGEXP, proxy)
    if use_proxy:
        with zipfile.ZipFile(CHROME_PLUGIN_FILE, 'w') as zp:
            zp.writestr("manifest.json", chrome_manifest())
            zp.writestr("background.js", chrome_background_js(
                host=use_proxy['host'], port=int(use_proxy['port']),
                user=use_proxy['usr'], password=use_proxy['pwd']))
        chrome_options.add_extension(CHROME_PLUGIN_FILE)
    if user_agent:
        chrome_options.add_argument('--user-agent=%s' % user_agent)
    return webdriver.Chrome(options=chrome_options, executable_path=get_latest_chromedriver_path())
