import importlib
import sys

from core.db import db_proxy
from peewee import *


class Proxy(Model):

    ip = IPField()
    port = IntegerField()
    user = CharField()
    password = CharField()
    scheme = CharField()

    country = CharField(max_length=64)

    is_active = BooleanField(default=True)

    class Meta:
        database = db_proxy

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.locked = False

    def __str__(self):
        return f"<Proxy {self.get_full_address()}>"

    def __repr__(self):
        return str(self)

    def lock(self):
        self.locked = True

    def unlock(self):
        self.locked = False

    def is_locked(self):
        return self.locked

    def get_full_address(self):
        return f"{self.scheme}://{self.user}:{self.password}@{self.ip}:{self.port}"


class Crawling(Model):

    name = CharField()
    script_name = CharField()

    class Meta:
        database = db_proxy

    def __str__(self):
        return f"<Crawling: {self.name}>"

    def __repr__(self):
        return str(self)

    def get_crawler(self):
        sys.path.append("core")
        sys.path.append("crawlers")
        crawler_module = importlib.import_module(f"crawlers.{self.script_name}")
        return getattr(crawler_module, "crawler")


class ProcessedNode(Model):

    name = CharField()
    ancestor_path = TextField()
    parent_id = CharField(max_length=32)

    crawling = ForeignKeyField(Crawling, backref='processed_nodes')

    class Meta:
        database = db_proxy

    @property
    def layer(self):
        return len(self.ancestor_path.split("/"))


class ErrorText(Model):

    text = TextField()

    class Meta:
        database = db_proxy


class ErrorNode(Model):

    DELIMITER = ";/;"

    name = CharField()
    path = TextField()
    url = CharField(max_length=1000)
    properties_chain = TextField()

    error_text = ForeignKeyField(ErrorText, backref="error_nodes")
    dom_file_path = CharField(max_length=512)
    screenshot_path = CharField(max_length=500)
    proxy = ForeignKeyField(Proxy, backref='errors')

    class Meta:
        database = db_proxy

    @property
    def layer(self):
        return len(self.path.split("/"))


